/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#ifndef ArmageTron_GAMEOBJECT_H
#define ArmageTron_GAMEOBJECT_H

#include "uInput.h"
#include "tList.h"
#include "eTess.h"

// a generic object for the game (cycles,explosions, particles,
// maybe AI opponents..)
class eGameObject{
  friend class eFace;
  friend class eCamera;
  friend class eSensor;


  // a list of all active gameobjects
  int id;
#define GO ((eGameObject *)NULL)
#define GO_I ((int *)NULL)

protected:
  // a list of all eGameObjects that are interesting to watch
  int interestingID;
  int inactiveID;

  // shall s_Timestep delete a eGameObject requesting destruction
  // completely (autodelete=1) or should it just be removed from the list
  // (autodelete=0) ?
  // (the latter may be useful if there exists other pointers to
  // the object)
  bool autodelete; 
  REAL lastTime;          // the time it was last updated
  REAL deathTime;          // the time the thing died

  eCoord pos;               // current position,
  eCoord dir;               // direction
  REAL  z;                 // and height (currently unused)

  tCHECKED_PTR(eFace) currentFace;  // the eFace pos it is currently

  // entry and deletion in the list of all eGameObjects
public:
  int team;                 // the team we belong to

  static uActionPlayer se_turnLeft,se_turnRight;

  static List<eGameObject> gameObjects;
  static List<eGameObject> gameObjectsInactive;
  static List<eGameObject> gameObjectsInteresting;

  void AddToList();
  void RemoveFromList();
  void RemoveFromListsAll();
  virtual void RemoveFromGame(); // call this instead of the destructor

  int GOID(){return id;}
  REAL LastTime(){return lastTime;}

  eGameObject(const eCoord &p,const eCoord &d, bool autodelete=1);
  virtual ~eGameObject();

  virtual eCoord Position()const{return pos;}
  virtual eCoord Direction()const{return dir;}
  virtual REAL DeathTime()const{return deathTime;}
  virtual REAL  Speed()const{return 20;}

  // position after FPS dependant extrapolation
  virtual eCoord PredictPosition(){return pos;} 

  // makes two gameObjects interact:
  virtual void InteractWith(eGameObject *target,REAL time,int recursion=1);

  // what happens if we pass eWall w? (at position e->p[0]*a + e->p[1]*(1-a) )
  virtual void PassEdge(const eEdge *e,REAL time,REAL a,int recursion=1);

  // moves the object from pos to dest during the timeinterval
  // [startTime,endTime] and issues all eWall-crossing tEvents
  void Move(const eCoord &dest,REAL startTime,REAL endTime);

  // finds the eFace we are in
  void FindCurrentFace();

  // simulates behaviour up to currentTime:
  virtual bool Timestep(REAL currentTime); 
  // return value: shall this object be destroyed?

  virtual bool EdgeIsDangerous(const eEdge *e, REAL time, REAL a){
    return e->Wall();
  }

  // destroys the gameobject (in the game)
  virtual void Kill();

  // tells whether the object is alive
  virtual bool Alive(){return false;}

  // draws it to the screen using OpenGL or PowerPack
  virtual void Render();

  // draws the cockpit or whatever is seen from the interior
  // in fixed perspective, called before the main rendering
  virtual bool RenderCockpitFixedBefore(bool primary=true);   
  // return value: draw everything else?

  // the same purpose, but called after main rendering
  virtual void RenderCockpitFixedAfter(bool primary=true);   

  // virtual perspective
  virtual void RenderCockpitVirtual(bool primary=false);

  //sound output
  virtual void SoundMix(Uint8 *dest,unsigned int len,
			int viewer,REAL rvol,REAL lvol){}; 

  // internal camera
  virtual eCoord CamDir(){return dir;}
  virtual REAL  CamRise(){return 0;}
  virtual eCoord CamPos() {return pos;}
  virtual REAL  CamZ(){return z;}
  virtual eCoord  CamTop(){return eCoord(0,0);}

  // sr_laggometer
  virtual REAL Lag() const{return 0;}


#ifdef POWERPAK_DEB
  virtual void PPDisplay();
#endif

  // Receives control from ePlayer
  virtual bool Act(uActionPlayer *Act,REAL x);
  
  // does a timestep and all interactions for every gameobject
  static void s_Timestep(REAL currentTime);
  // does a timestep and all interactions for this gameobject,
  // divided in many small steps
  static bool TimestepThis(REAL currentTime,eGameObject *t);

  // displays everything:
  static void RenderAll();
  static void PPDisplayAll();

  // kills everything:
  static void DeleteAll();
};

#endif
