/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#include "eTess.h"
#include "rTexture.h"
#include "tEventQueue.h"
#include "eTimer.h"
#include "eWall.h"
#include "eGameObject.h"
#include "eCamera.h"

#include <math.h>

#ifdef _MSC_VER
#include <float.h>
#define finite _finite
#endif

int se_debugExt=0;
 
List<ePoint> ePoint::points;

List<eEdge> eEdge::edges;

List<eFace> eFace::faces;
int        eFace::s_currentCheckVis[MAX_VIEWERS];


// ********************************************

// consistency checks:

void ePoint::Check(bool lp){
  int j;
  for(int i=points.Len()-1;i>=0;i--){
    ePoint *p=points(i);

    // check for loose points:
    if (lp && p->refCtr+p->f.Len()==0)
      tERR_ERROR_INT("Loose point!");

    // check if the point is contained in all its neighbouring faces:
    for(j=p->f.Len()-1;j>=0;j--){
      eFace *F=p->f(j);
      if (F->FindPoint(p)<0)
	tERR_ERROR_INT("Not a corner of neighbouring eFace!");
    }
    
    // check for wrong refCtr
    /*
    if (p->refCtr){
      int real_refCtr=0;
      for(j=eEdge::edges.Len()-1;j>=0;j--){
	eEdge *e=eEdge::edges(j);
	if (!e->f[0]){
	  if (e->p[0]==p)
	    real_refCtr++;
	  if (e->p[1]==p)
	    real_refCtr++;
	}
      }
      if (p->refCtr!=real_refCtr)
	tERR_ERROR_INT("wrong refCtr!");
    }
    */
  }
}

void eEdge::Check(){
  for(int i=edges.Len()-1;i>=0;i--){
    eEdge *e=edges(i);

    // check for null lines
    if (e->p[0]==e->p[1])
      tERR_ERROR_INT("Null line!");

    // check for meaningless isolated lines
    if (!e->wall && *e->p[0]==*e->p[1])
      tERR_ERROR_INT("isolated line!");

    // check for isolated eWalls with incorrect refCtr on the points
    if (!e->f[0] && (e->p[0]->refCtr < 1 || e->p[1]->refCtr < 1))
      tERR_ERROR_INT("wrong refCtr!");

    // check if the eEdge is contained in all its neighbouring faces:
    for(int j=1;j>=0;j--){
      eFace *F=e->f[j];
      if (F && F->FindEdge(e)<0)
	tERR_ERROR_INT("Not an eEdge of neighbouring eFace!");
    }
  }
}

void eFace::Check(){

}

void se_CheckGrid(bool lp){
  ePoint::Check(lp);
  eEdge::Check();
  eFace::Check();
}



/* **********************************************************
   Points:
   ********************************************************** */

#ifdef POWERPAK_DEB
void ePoint::DebugPlot(PD_Color c){
  if (pp_tess_deb && se_debugExt>=0){
    PD_PutPixel (DoubleBuffer,se_X_ToScreen(x), se_Y_ToScreen(y), c);
    if (se_debugExt) PD_ShowDoubleBuffer();
  }
}
#endif

ePoint::ePoint(REAL X,REAL Y):eCoord(X,Y),id(-1){
#ifdef DEBUG
  if (!finite(x) || !finite(y))
    st_Breakpoint();
#endif

  se_GridRange(Norm_squared());
  refCtr=0;
  points.Add(this,id);

#ifdef POWERPAK_DEB
  if (pp_out) DebugPlot(PD_CreateColor(DoubleBuffer,0xFF,0xFF,0xFF));
#endif
}

ePoint::ePoint(const eCoord &c):eCoord(c),id(-1){
#ifdef DEBUG
  if (!finite(x) || !finite(y))
    st_Breakpoint();
#endif

  refCtr=0;
  points.Add(this,id);

#ifdef POWERPAK_DEB
  if (pp_out) DebugPlot(PD_CreateColor(DoubleBuffer,0xFF,0xFF,0xFF));
#endif
}

ePoint::ePoint():id(-1){
#ifdef DEBUG
  if (!finite(x) || !finite(y))
    st_Breakpoint();
#endif

  refCtr=0;
#ifdef POWERPAK_DEB
  if (pp_out) DebugPlot(PD_CreateColor(DoubleBuffer,0x00,0x00,0x00));
#endif

 if (f.Len()>0)
   tERR_ERROR_INT("Destroying ePoint with neighbours!");
}

ePoint::~ePoint(){
#ifdef DEBUG
  if (refCtr)
    tERR_ERROR_INT("Point still has " << refCtr << " references!");

  if (f.Len())
    tERR_ERROR_INT("Point still is member of " << f.Len() << " faces!");
#endif

  points.Remove(this,id);
  tCHECK_DEST;
}

void eCoord::Print(ostream &s)const{ s << "(" << x << "," << y << ")"; }
void eEdge::Print(ostream &s)const{ s << "[" << *p[0] << "->" << *p[1] << "]"; }
void eFace::Print(ostream &s)const{ s << "[" << *p[0] << "->" << *p[1] << "->" << *p[2] << "]"; }


void ePoint::AddNeighbour(eFace *F){

#ifdef DEBUG
  int good=0;
  int i;
  for(i=0;i<=2;i++)
    if (F->p[i]==this) good=1;

  if(!good) tERR_ERROR_INT("Face " << *F << " does not contain ePoint " << *this << '!');

  for(i=f.Len()-1;i>=0;i--)
    if (f(i)==F) tERR_ERROR_INT("Face " << *F << " allready marked as neighbour of"
		       " ePoint " << *this << '!');
#endif
  
  f[f.Len()]=F;
}

void ePoint::RemoveNeighbour(eFace *F){
  int num=-1;
  for(int i=f.Len()-1;i>=0;i--){
    if (f(i)==F) 
      num=i;
    }

  if (num==-1){
    tERR_WARN("Face " << *F << " not marked as neighbour of"
	      " ePoint " << *this << '!');
  }
  else{
    f(num)=f(f.Len()-1);
    f.SetLen(f.Len()-1);
  }
}


void ePoint::Replace(ePoint *P){
  int i,j,k;

  if (this!=P){
    if (!(*this==*P))
      tERR_WARN("Unequal points are going to get exchaned!");

    *this=*P;

    while(f.Len()){
      eFace *F=f(0);
      RemoveNeighbour(F);
      
      for(j=0;j<=2;j++){
	if(F->p[j]==this) 
	  F->p[j]=P;
	
	for(k=0;k<=1;k++)
	  if (F->e[j]->p[k]==this){
	    F->e[j]->p[k]=P;
	    F->e[j]->calc_Len();
	  }
	
      }
      P->AddNeighbour(F);
    }
  
    if (refCtr){
      // shit! we need to go through all lines...
      
      for(i=eEdge::edges.Len()-1;i>=0;i--)
	for(k=1;k>=0;k--)
	  if (eEdge::edges(i)->p[k]==this){
	    eEdge::edges(i)->p[k]=P;
	    P->refCtr++;
	    refCtr--;
#ifdef DEBUG
	    if (eEdge::edges(i)->f[0])
	      tERR_ERROR_INT("Unexpected neighbouring eFace!");
#endif
	  }
#ifdef DEBUG
    if (refCtr)
      tERR_ERROR_INT("Refcount was wrong!");
#endif
    }
  }
}


ePoint * ePoint::DrawLineTo(ePoint *end,eWall *wal, bool change_grid){
  se_GridRange(end->Norm_squared());
  int restart=1;
  ePoint *start=this;
  eEdge *todo=tNEW(eEdge) (start,end,wal);

  // now the loop: go from source to destination, 
  // "rotate" the edges we cross out of our way (if possible)
  // or cut them in half.

  eFace* current=NULL; // the eFace the line is about to cut in half
  int   ePoint_in=0; // the ePoint it came in; we will leave through the
  // opposite side (if we leave at all..)

  eCoord dir=*end-*start; // the direction we are going

  REAL left_test,right_test;
  bool exactly_onLeft=0;

  bool goon=1;
  while (goon && dir.Norm_squared()>0){

    // first, we need to find the first eFace:
    int i;
    
    if (restart){
      if (!start->f.Len()) 
	start->InsertIntoGrid();
      for(i=start->f.Len()-1;i>=0;i--){
	eFace *test=start->f(i);
	
	ePoint_in=test->FindPoint(start);
	if(ePoint_in==-1) 
	  tERR_ERROR_INT("Point " << *start << 
		" is not member of neighbouring eFace "
		<< *test << '!');
	
	
	// the two eEdge vectors ePointing away from *start in
	// eFace test
	eCoord lvec=test->LeftVec(ePoint_in);
	eCoord rvec=test->RightVec(ePoint_in);
	
	// is dir between lvec and rvec?
	left_test=lvec*dir;
	right_test=rvec*dir;
	
	if (left_test == 0 && right_test == 0)
	  {
	    st_Breakpoint();
	  }

	
	if (left_test >=0 && right_test < 0){
	  
	  // are dir and lvec exactly on one line?
	  exactly_onLeft=(left_test<=EPS*se_EstimatedRangeOfMult(lvec,dir)); 
	  
	  // if (left_test<=0) con << "\nACHTUNG!!!!!\n\n";
	  current=test;
	  i=-1;
	}
      }
      if (!current)
	tERR_ERROR("Point " << *start << " does not have a eFace in direction " 
		   << dir << "! I know this bug happens a lot more often than"
              " I'd like to. Don't send a bug report!");
      
      restart=0;
    }

    if (start!=current->p[ePoint_in])
      tERR_ERROR_INT("ePoint_in not correct!");
    
    // are we finished?
    eCoord left_ePoint_to_dest=-(*(current->p[se_Left(ePoint_in)])-*end);
    eCoord lvec=current->LeftVec(se_Left(ePoint_in));
    
    REAL test=lvec*left_ePoint_to_dest;
    if(test>0){
      // endpoint lies within current eFace; cut it into three and be finnished.
      goon=0;
      
      /*
	
        C        a        B
	................
	.  .*cc     *** .
       .  . *  D    bb.
	   .  |      .
    e1	   .  |     .
	    . |aa  .
   .      b . |   .  c
 ..   e3     .|  .
E         G  .| .
    .	      ..
      e2      .

	      A

       */
      
      ePoint *A=current->p[ePoint_in];
      
      if ((*A)==*end){
	A->Replace(end);
	delete A;
#ifdef DEBUG
  se_CheckGrid();
#endif
	return end;
      }
      else{
	todo->MakeFirst(A);
	if ((!(todo->p[0]==A)) || (!(todo->p[1]==end)))
	  tERR_ERROR_INT ("Arg!");

	ePoint *B=current->p[se_Right(ePoint_in)];
	ePoint *D=end;
	ePoint *C=current->p[se_Left(ePoint_in)];
	eEdge  *a=current->e[ePoint_in];
	eEdge  *b=current->e[se_Right(ePoint_in)];
	eEdge  *c=current->e[se_Left(ePoint_in)];
	
	if(!exactly_onLeft){
	  // no problem: D really lies within current eFace.
	  // just split it in three parts:
	  
	  delete current;
	  current=0;
	  
	  eEdge *bb=tNEW(eEdge) (B,D);
	  eEdge *cc=tNEW(eEdge) (C,D);
	  tNEW(eFace) (b,todo,cc);
	  tNEW(eFace) (bb,todo,c);
	  tNEW(eFace) (bb,a,cc);
	  
#ifdef DEBUG
  se_CheckGrid();
#endif
	  return D;
	}
	else{
	  // bad luck: D lies on eEdge b. We are going to need the
	  // other triangle touching b:
	  eFace *G=b->Other(current);
	  
	  if (G){
	    int eEdge_b=G->FindEdge(b);
	    
	    
#ifdef DEBUG
	    // paranoia:
	    if(eEdge_b<0)
	      tERR_ERROR_INT("Face " << *G << " does not contain eEdge " << *b << '!');
	    if (G->p[se_Left(eEdge_b)]!=C)
	      tERR_ERROR_INT("Face " << *G << " does not contain ePoint " << *C << '!');
	    if (G->p[se_Right(eEdge_b)]!=A)
	      tERR_ERROR_INT("Face " << *G << " does not contain ePoint " << *A << '!');
#endif
	    
	    ePoint *E=G->p[eEdge_b];
	    eEdge *e1=G->e[se_Right(eEdge_b)];
	    eEdge *e2=G->e[se_Left(eEdge_b)];
	    
	    
	    // delete faces inside quad ABCE and connect A...E with new
	    // ePoint D.

	    delete G;
	    delete current;
	    delete b;
	    
	    eEdge *DE=tNEW(eEdge) (D,E);
	    eEdge *DB=tNEW(eEdge) (D,B);
	    eEdge *DC=tNEW(eEdge) (D,C);
	    
	    tNEW(eFace) (DE,DC,e1);
	    tNEW(eFace) (DC,DB,a);
	    tNEW(eFace) (DB,todo,c);
	    tNEW(eFace) (todo,DE,e2);

#ifdef DEBUG
  se_CheckGrid();
#endif
	    return D;
	  }
	  else{
	    // better luck: no G!

	    delete current;
	    delete b;
	    
	    eEdge *DB=tNEW(eEdge) (D,B);
	    eEdge *DC=tNEW(eEdge) (D,C);
	    
	    tNEW(eFace) (DC,DB,a);
	    tNEW(eFace) (DB,todo,c);

#ifdef DEBUG
  se_CheckGrid();
#endif
	    return D;
	  }
	  
	}
      }
      // finnished!
    }
    else{
      // some definitions and trivial tests:

      eFace *F=current;
      ePoint *C=F->p[ePoint_in];
      if (*C==*end){
	C->Replace(end);
	delete C;

#ifdef DEBUG
  se_CheckGrid();
#endif
	return end;
      }
      eEdge *dummy=NULL;

      ePoint *D=F->p[se_Left(ePoint_in)];
      eEdge  *d=F->e[se_Right(ePoint_in)];
      if (*D==*end){
	todo->Split(d,dummy);
	delete dummy;
	delete todo;
	D->Replace(end);
	delete D;

#ifdef DEBUG
  se_CheckGrid();
#endif
	return end;
      }

      ePoint *B=F->p[se_Right(ePoint_in)];
      eEdge  *c=F->e[se_Left(ePoint_in)];
      if (*B==*end){
	todo->Split(c,dummy);
	delete dummy;
	delete todo;
	B->Replace(end);
	delete B;

#ifdef DEBUG
  se_CheckGrid();
#endif
	return end;
      }

      // if dir is parallel with eEdge c:
      if(exactly_onLeft){
	/*
	con << "Parallel!\n";
	if (pp_out)
	  PD_ShowDoubleBuffer();
	se_debugExt=1;
	*/
	todo->Split(d,dummy);
	delete todo;
	todo=dummy;
	start=todo->p[0];
	restart=1;
      }
      
      else{
	eEdge  *e=F->e[ePoint_in];
	eFace  *G=e->Other(F);
	
	if (G==NULL){
	  tERR_ERROR_INT ("Left Grid!");
	}
	else{
	  
	  // what number does e have in G?
	  
	  int eEdge_in=G->FindEdge(e);
	  
	  // safety checks
#ifdef DEBUG
	  if (eEdge_in==-1) tERR_ERROR_INT("Edge " << *e << " does not belong to Face "
				 << *G << '!');
	  if (D!=G->p[se_Right(eEdge_in)]) tERR_ERROR_INT("Faces " << *F << " and " << *G 
					     << " do not fit!");
	  if (B!=G->p[se_Left(eEdge_in)]) tERR_ERROR_INT("Faces " << *F << " and " << *G 
					    << " do not fit!");
#endif
	  
	  eEdge *b=G->e[se_Right(eEdge_in)];
	  eEdge *a=G->e[se_Left(eEdge_in)];
	  ePoint *A=G->p[eEdge_in];
	
	/* we have the following Situation:
	   
	   
                A      
	        *
               * **
	      *    ** b
	    a*       **
	    *   G |    **
	   *      |   e  ** 
	  D********|********* B
	    *      |(new eWall)
	     *   F |      *
	      *    |     *
	       *   |    *
	     d  *  |   * c 
	         * |  *
	          *| *
	           **
	           C
             
      */
      
      /*

	if the angles CDA and ABC are less than 180deg, we can just 
	"turn" eEdge e:


           A      
            *
	   ****
          * *  ** b
        a*  *    **
        *   * |    **
       *    e*|  G   ** 
     D*   F  *|       ** B
       *     *|(new eWall)
        *    *|      *
         *   *|     *
          *  *|    *
        d  *  *   * c 
            * *  *
	     ****
              **
	      C


      */

	  eCoord dd=*C-*D;
	  eCoord aa=*A-*D;
	  eCoord bb=*A-*B;
	  eCoord cc=*C-*B;
	  
	  REAL scale = aa.Norm_squared() + bb.Norm_squared() +
	    bb.Norm_squared() + cc.Norm_squared();
			    
	  if (change_grid && e->Movable() && aa*dd>scale*.0001 && cc*bb >scale*.0001){

	    delete F;
	    delete G;
	    delete e;
	    
	    e=tNEW(eEdge) (A,C);
	    F=tNEW(eFace) (a,d,e);
	    G=tNEW(eFace) (b,e,c);
	    
	    ePoint_in=0;
	    eCoord ee=*A-*C;
	    REAL x_test=ee*dir;
	    if (x_test>=0){
	      exactly_onLeft=(x_test<=EPS*se_EstimatedRangeOfMult(ee,dir));
	      current=G;
	    }
	    else{
	      exactly_onLeft=0;
	      current=F;
	    }
	  }
	  
	  /* if not, we hace to introduce a new ePoint E: the cut between our
	     new eWall and eEdge e.
	     
	     A      
	     *
	   * **
          *  * ** b
        a*   a'  **
        *  G *| G' **
       *      *E     ** 
     D***d`*******b`**** B
       *      *(new eWall)
        *  F  *      *
         *    * F'  *
          *   *    *
        d  *  c'  * c 
            * *  *
	     ****
              **
	      C


     */

	  else{
	    // calculate the coordinates of the intersection ePoint E:

	    ePoint *E=e->IntersectWithCareless(todo);

	    delete F;
	    delete G;

	    eEdge *bb,*cc,*dd,*newtodo;
	    e->SplitOriented(dd,bb,E,D);
	    delete e;

	    todo->SplitOriented(cc,newtodo,E,C);
	    delete todo;
	    todo=newtodo;
	    start=todo->p[0];

	    eEdge *aa=tNEW(eEdge) (E,A);
	    
	    F=tNEW(eFace) (d,cc,dd);
	    G=tNEW(eFace) (a,dd,aa);
	    eFace *GG=tNEW(eFace) (b,aa,bb);
	    tNEW(eFace) (c,bb,cc);

	    ePoint_in=0;
	    
	    eCoord ea=*A-*E;
	    REAL x_test=ea*dir;
	    if (x_test>=0){
	      current=GG;
	      exactly_onLeft=(x_test<=EPS*se_EstimatedRangeOfMult(ea,dir));
	      // if (exactly_onLeft) cerr << "Achtung 3!\n";
	    }
	    
	    else{
	      current=G;
	      exactly_onLeft=0;
	    }
	  }
	}
      }
    }
  }
  tERR_ERROR_INT("We shound never get here!");
  return NULL;
}

ePoint * ePoint::InsertIntoGrid(eCoord x){
#ifdef DEBUG
  se_CheckGrid();
#endif
  se_GridRange(x.Norm_squared());

  for(int i=eFace::faces.Len()-1;i>=0;i--)
    if (eFace::faces(i)->IsInside(x)){
      for(int j=0;j<=2;j++){
	if (*eFace::faces(i)->p[j]==x)
	  return eFace::faces(i)->p[j];

      }
      return eFace::faces(i)->p[0]->DrawLineTo(x);
      break;
    }
  return NULL; 
}

void ePoint::InsertIntoGrid(){
  if (f.Len()==0){
    for(int i=eFace::faces.Len()-1;i>=0;i--)
      
      if (eFace::faces(i)->IsInside(*this)){
	for(int j=0;j<=2;j++){
	  if (*eFace::faces(i)->p[j]==*this){
	    ePoint *same=eFace::faces(i)->p[j];
	    same->Replace(this);
	    delete same;
	    return;
	  }
	}
	eFace::faces(i)->p[0]->DrawLineTo(this);
	return;
      }
  }
}

void ePoint::Clear(){
  for (int i=points.Len()-1;i>=0;i--)
    delete points[i];
}


/* **********************************************************
   Edge-Crossings:
   ********************************************************** */


eEdgeViewer **eEdgeViewer::Ref(){
  if(!e)
    return NULL;
  else{
    eEdgeViewer **ret=&e->edgeViewers;
    while (*ret){
      if ((*ret)==this)
	return ret;
      ret=&((*ret)->next);
    }
    tERR_ERROR_INT("eEdgeViewer does not belong to eEdge!");
  }
  
}

eEdgeViewer::eEdgeViewer(eEdge *E,int v)
  :next(E->edgeViewers),e(E),viewer(v){
  E->edgeViewers=this;
}
  
eEdgeViewer::~eEdgeViewer(){
  eEdgeViewer **r=Ref();
  if (r)
    *r = next;
  tCHECK_DEST;
}

// *********************************************************

tEventQueue eEdge_crossing[MAX_VIEWERS];

// tEvents for a viewer crossing an eEdge (that is, the staight
// prolongiation of the eEdge):

eViewerCrossesEdge::eViewerCrossesEdge(eEdge *E,int v)
  :eEdgeViewer(E,v){
  eEdge_crossing[v].Insert(this);
}
  
eViewerCrossesEdge::~eViewerCrossesEdge(){
  if (hID>=0)
    eEdge_crossing[viewer].Remove(this);
  tCHECK_DEST;
}

bool eViewerCrossesEdge::Check(REAL dist){
  if (e->UpdateVis(viewer)){
    eCoord p0_to_viewer=(CameraPos(viewer) - *e->p[0]);
    eCoord evec=*e->p[1] - *e->p[0];
    value=dist + fabs((p0_to_viewer*evec))/e->len;
    return true;
  }
  else
    return false;
}

tHeapBase *eViewerCrossesEdge::Heap(){
  return &eEdge_crossing[viewer];
}



/* **********************************************************
   Edges:
   ********************************************************** */


#ifdef POWERPAK_DEB
void eEdge::DebugPlot(PD_Color c){
  if (pp_tess_deb && se_debugExt>=0){
    PD_Line (DoubleBuffer, se_X_ToScreen(p[0]->x), se_Y_ToScreen(p[0]->y),
	     se_X_ToScreen(p[1]->x), se_Y_ToScreen(p[1]->y), c);   
    if (se_debugExt) PD_ShowDoubleBuffer();
  }
}

void eEdge::DebugPlotRefresh(){
  PD_Color c;
  if (pp_out){
    if (!wall) c=PD_CreateColor(DoubleBuffer, 0x0, 0x80, 0x80);
    else c=PD_CreateColor(DoubleBuffer, 0x00, 0xFF, 0x00);
    DebugPlot(c);
  }
}
#endif

void eEdge::calc_Len(){
  len=(*p[0] - *p[1]).Norm_squared();
  len=sqrt(len);
}

eEdge::eEdge(ePoint *p1,ePoint *p2,eWall *W):edgeViewers(NULL){
  p[0]=p1;p[1]=p2;f[0]=NULL;f[1]=NULL;wall=W;id=-1;
  
  if (p1 == p2)
    tERR_ERROR_INT ("Null line!");

  p[0]->AddRef();
  p[1]->AddRef();

  if (wall){
    for(int i=MAX_VIEWERS-1;i>=0;i--)
      wall->SetVisHeight(i,-1);
#ifdef DEBUG
    if (wall->edge)
      tERR_ERROR_INT("Hey! Wall has already an eEdge!");
#endif
    wall->edge=this;
  }
  
  calc_Len();

  edges.Add(this,id);

  //con << "Creating eEdge " << id << "\n";

#ifdef POWERPAK_DEB
  DebugPlotRefresh();
#endif

  if (wall)
    wall->Insert();
}



eEdge::~eEdge(){
  //con << "Killing eEdge " << id << "\n";
  edges.Remove(this,id);

  while(edgeViewers) 
    delete edgeViewers;

  if (f[0])
    tERR_ERROR_INT("Hey! you can't delete this eEdge! It is connected to eFace "
	  << *f[0] << '!');
  
  if(p[0])
    p[0]->Release();
  if (p[1])
    p[1]->Release();

  tDESTROY(wall);

#ifdef POWERPAK_DEB
  if (pp_out) DebugPlot(PD_CreateColor(DoubleBuffer,0,0,0));
#endif
  tCHECK_DEST;
}

void eEdge::SetWall(eWall *w){
  tDESTROY(wall);
#ifdef DEBUG
  if (w->edge)
    tERR_ERROR_INT("Hey! Wall has already an eEdge!");
#endif
  w->edge=this;
  wall=w;

  if (w)
    w->Insert();
}


void eEdge::AddNeighbour(eFace *F){

#ifdef DEBUG
  int good=0,i;
  for(i=0;i<=2;i++)
    if (F->e[i]==this) good=1;

  if(!good) tERR_ERROR_INT("Face " << *F << " does not contain eEdge " << *this << '!');

  for(i=1;i>=0;i--)
    if (f[i]==F) tERR_ERROR_INT("Face " << *F << " allready marked as neighbour of"
		       " eEdge " << *this << '!');
#endif
  
  if (f[0]==NULL){
    f[0]=F;
    p[0]->Release();
    p[1]->Release();
  }
  else if (f[1]==NULL)
    f[1]=F;
  else tERR_ERROR_INT ("Edge " << *this << " allready has two neighbours!");
}

void eEdge::RemoveNeighbour(eFace *F){
  int num=-1;
  for(int i=1;i>=0;i--)
    if (f[i]==F) num=i;
  
  if (num==-1){
    tERR_ERROR_INT("Face " << *F << " not marked as neighbour of"
	  " eEdge " << *this << '!');
  }
  else{
    if (num==0){
      f[0]=f[1];
      if (!f[0]){
	p[0]->AddRef();
	p[1]->AddRef();
      }
    }
    f[1]=NULL;
  }
}

void eEdge::Flip(){
  if (wall) wall->Flip();
  Swap(p[0],p[1]);
}

bool eEdge::Splittable() const{return !wall || wall->Splittable();}

void eEdge::Split(eEdge *& e1,eEdge *& e2,ePoint *s){
  
  if (!wall){ // ha! simple.
    e1=tNEW(eEdge) (p[0],s);
    e2=tNEW(eEdge) (s,p[1]);
  }
  else{
    if (!wall->Splittable())
      tERR_ERROR_INT("I told you not to split that!");
    // first, split the eWall structure:
    eWall *w1,*w2;    
    wall->Split(w1,w2,Ratio(*s));

    // then, create two new eWalls with the split structure.
    e1=tNEW(eEdge) (p[0],s,w1);
    e2=tNEW(eEdge) (s,p[1],w2);
  }
}

void eEdge::Split(eEdge * e,eEdge *& e1){
  eEdge *e2;
  if (p[0]==e->p[1] || p[1]==e->p[1])
    e->Flip();
  if (p[1]==e->p[0])
    Flip();

  if (!(p[0]==e->p[0]))
    tERR_ERROR_INT("No common endpoint!");

  Split(e2,e1,e->p[1]);
  eWall *w=e2->wall;
  w->edge=NULL;
  e->SetWall(w);
  e2->wall=0;
  delete e2;
}

void eEdge::SplitOriented(eEdge *& e1,eEdge *& e2,ePoint *s,ePoint *first){
  MakeFirst(first);
  Split(e1,e2,s);
}

ePoint * eEdge::IntersectWithCareless(const eEdge *e2) const{
  eCoord vec  = *p[1]     -     *p[0];
  eCoord e2vec= *e2->p[1] - *e2->p[0];
  eCoord e2_t = *p[0]     - *e2->p[0];

  REAL inv = (vec*e2vec);
  if (inv!=0)
    return tNEW(ePoint) (*p[0]-vec*((e2_t*e2vec)/inv));
  else
    return tNEW(ePoint) (*p[0]-vec*((e2_t*e2vec)));
}  

ePoint * eEdge::IntersectWith(const eEdge *e2) const{
  eCoord vec  = *p[1]     -     *p[0];
  eCoord e2vec= *e2->p[1] - *e2->p[0];
  eCoord e2_t = *p[0]     - *e2->p[0];
  
  REAL div=(vec*e2vec);

  if (div==0) return NULL;

  eCoord ret=*p[0]-vec*((e2_t*e2vec)/ div);


  REAL v=Ratio(ret);
  if (v<0 || v>1) return NULL;
  v=e2->Ratio(ret);
  if (v<0 || v>1) return NULL;

#ifdef POWERPAK_DEB
  if (pp_tess_deb)
    PD_CircleFilled (DoubleBuffer,
		     se_X_ToScreen(ret.x), se_Y_ToScreen(ret.y),
		     5,1,PD_CreateColor(DoubleBuffer,255,255,255));
#endif

  return tNEW(ePoint) (ret);
}  

void eEdge::CopyIntoGrid(bool change_grid){
  calc_Len();
  p[0]->InsertIntoGrid();
  eWall *w=wall;
  w->edge=NULL;
  wall=0;
#ifdef DEBUG
  //  se_CheckGrid();
#endif
  p[0]->DrawLineTo(p[1],w,change_grid);
#ifdef DEBUG
  //  se_CheckGrid();
#endif
}

void eEdge::MarkCheckVis(int i){
  eEdgeViewer *x=edgeViewers;

  while (x){
    if (x->viewer==i){
      delete x;
      x=edgeViewers;
    }
    if (x) x=x->next;
  }

  if (i<NumberOfCameras())
    tNEW(eViewerCrossesEdge) (this,i);
}


eFace * eEdge::NearerToViewer(int i){
  if (f[0]){
    ePoint * opposite=f[0]->p[f[0]->FindEdge(this)]; 
    // the ePoint opposite to this eEdge in eFace 0.

    eCoord first_to_opposite=*opposite - *p[0];
    // the vector from ePoint 0 to the opposite ePoint

    eCoord first_to_viewer=CameraPos(i) - *p[0];
    // the vector from ePoint 0 to the viewer

    eCoord first_to_second=*p[1] - *p[0];
    // the eEdge vector

    // are viewer and opposite ePoint on the same side of the
    // straigth througt our eEdge? (has their product the same sign?)
    if ( (first_to_viewer*first_to_second) 
	*(first_to_opposite*first_to_second) > 0)
      // yes! eFace 0 is nearer.
      return f[0];
    else
      // no! eFace 1 is nearer.
      return f[1];
  }
  else return NULL;
}


bool eEdge::UpdateVis(int i){
  // eFace nearer to the viewer:
  eFace *nearer=NearerToViewer(i);
  if (nearer){
    eFace *farther=Other(nearer);
    if (farther){
      // we have a far and a near side.
      // we can see that low on the near side:
      REAL nh=nearer->visHeight[i];
      REAL fh=farther->visHeight[i];

      if (wall) wall->SetVisHeight(i,nh);

      if (nh<fh){ // possible visibility flow
	REAL h=nh;
	if (wall){
	  // the eWall blocks the vision more:
	  if (wall->BlockHeight()>h) h=wall->BlockHeight();
	}
	if (h<fh){ // visibility gets extended
	  farther->SetVisHeight(i,h);
	}
      }
      else if (fh<nh){ // maybe the near eFace needs an update?
	farther->UpdateVis(i);
	return true; // check again ( i am to layzy to make
	// a full analysis here; too much may have changed.)
      }
    }
  }
  return false;
}

void eEdge::UpdateVisAll(int i){
  eEdge_crossing[i].Timestep(se_cameras[i]->Dist());
}

void eEdge::UpdateVisAll(){
  for(int i=NumberOfCameras()-1;i>=0;i--)
    eEdge_crossing[i].Timestep(se_cameras[i]->Dist());
}

void eEdge::SeethroughHasChanged(){
  for (int i=edges.Len()-1;i>=0;i--){
    eEdge *e=edges(i);
    if (e->wall)
      for(int j=NumberOfCameras()-1;j>=0;j--){
	if(e->UpdateVis(j))
	  e->MarkCheckVis(j);
      }
  }
}

void eEdge::Clear(){
  for (int i=edges.Len()-1;i>=0;i--)
    delete edges[i];
}


bool eEdge::Simplify(int dir){
  calc_Len();

  if (wall && wall->Deletable() && f[0]){
    tDESTROY(wall);
    if (!f[0] && *p[0]==*p[1]){
      /*
	for(int k=1;k>=0;k--)
	if (p[k]->refCtr+p[k]->f.Len()==0)
	tDESTROY p[k]
      */
      return true;
    }
#ifdef DEBUG
    se_CheckGrid();
#endif
  }

  if (wall || (!f[0] && !f[1]) || p[dir]->refCtr)
    return false;

  int i;

  ePoint  *to_be_removed=p[dir]; // we'll remove this ePoint
  ePoint  *stay=p[1-dir];        // and redirect all his edges to this ePoint

  // first, we find the faces that are going to be modified:
  // they are neighbours of the ePoint we'll remove.

  for(i=to_be_removed->f.Len()-1;i>=0;i--){
    eFace *F=to_be_removed->f(i); // eFace to be modified
    int rem_ind=F->FindPoint(to_be_removed); // idex of our to-be-removed 
    // ePoint in that eFace

    // if any of the to-be-modified edges is a eWall, abort the operation.
    if (F->e[se_Left(rem_ind)]->wall)
      return false;

    // if the eFace is to be modified (and not deleted), check
    // if it will flip orientation; if yes, abort.

    if (F!=f[0] && F!=f[1]){
      eCoord leftvec =F->LeftVec (rem_ind);
      eCoord rightvec=F->RightVec(rem_ind);
      
      REAL or_before=leftvec*rightvec;

      // modify the vectors like they'll be after the modification
      leftvec =leftvec  + *to_be_removed - *stay;
      rightvec=rightvec + *to_be_removed - *stay;

      REAL or_after=leftvec*rightvec;
      
      // if the orientation changed, quit:
      if (or_after*or_before<=0)
	return false;
    }
  }

#ifdef DEBUG
  se_CheckGrid();
#endif
  
  // Allright! everything is safe.

  int side;

  for(side=1;side>=0;side--)
    if (f[side])
      {
	int rem_ind=f[side]->FindPoint(to_be_removed);
	int stay_ind=f[side]->FindPoint(stay);

	eEdge *g=f[side]->e[rem_ind];
	eEdge *e=f[side]->e[stay_ind];
	eFace *F=e->Other(f[side]);

#ifdef DEBUG
  if (to_be_removed->refCtr>0)
    tERR_ERROR_INT("unexpected reference!");
#endif

	// delete the eFace
	delete f[side];
	//f[side]=NULL;

#ifdef DEBUG
  if (to_be_removed->refCtr>2-side)
    tERR_ERROR_INT("unexpected reference!");
#endif

	if (F){
	  int e_rem_ind=F->FindEdge(e);

	      /* all right. Situation:
		 S: stay
		 R: to_be_removed
		 
		 
	              . *---------------*
		    .   |             .
	          .     |e          .
	       g.       |         .
              .         |    F  .
            .           |     .
          .    f[side]  |   .
        .               | .
      S-----------------R

      where the eEdge e is f[side]->e[stay_ind] as well as
      F->e[e_rem_ind], and g is f[side]->e[rem_ind].
     
      All we need to to (since we want to remove f[side], e and ultimately
      R) is to replace e in F's data structure by g:    */

	  F->e[e_rem_ind]=g;
	  g->AddNeighbour(F);
	  e->RemoveNeighbour(F);
	
	  // delete e and the only other reference to it:
	  
#ifdef DEBUG
	  if(e->f[0])
	    tERR_ERROR_INT("Unexpected non-NULL-pointer!");
#endif
	}
	delete e;
	e=NULL;
#ifdef DEBUG
	if (to_be_removed->refCtr>1-side)
	  tERR_ERROR_INT("unexpected reference!");
#endif
      }
  

#ifdef DEBUG
  if (to_be_removed->refCtr>1)
    tERR_ERROR_INT("unexpected reference!");
#endif

#ifdef DEBUG
  se_CheckGrid();
#endif

  return true;
}

void eEdge::SimplifyNum(int e,int dir){
#ifdef DEBUG
  se_CheckGrid();
#endif
  
  if (e>=0 && e < edges.Len()){
    eEdge *E=edges(e);
#ifdef DEBUG
    eEdge *Esave=E;
#endif
    if (E->Simplify(dir)){
      // remove all references to the ePoint we'll remove:
      ePoint *p=E->p[dir];
      ePoint *sp=E->p[1-dir];
      
      //#ifdef DEBUG
      //se_CheckGrid();
      //#endif
      
      tDESTROY_PTR(E);
      
#ifdef DEBUG
      se_CheckGrid(false);
#endif
      
      *p=*sp;
      p->Replace(sp);
      
#ifdef DEBUG
      se_CheckGrid(false);
#endif
      
      tDESTROY_PTR(p);
      
      // if sp is a loose ePoint, kill it:
      if (sp->refCtr + sp->f.Len() == 0){
#ifdef DEBUG
	con << "Warning removing eEdge " << Esave << ": sp should not be loose!\n";
#endif
	tDESTROY_PTR(sp);
      }

#ifdef DEBUG
      se_CheckGrid();
#endif
      
    }
  }
#ifdef DEBUG
  se_CheckGrid();
#endif
  
}

void eEdge::SimplifyAll(int count){
  static int counter=0;
  
  for(;count>0;count--){
    counter--;
    
    if (counter<0 || counter>=edges.Len()) 
      counter=edges.Len()-1;

      SimplifyNum(counter,0);
      SimplifyNum(counter,1);
  }
}


/* **********************************************************
   Faces:
   ********************************************************** */

#ifdef POWERPAK_DEB
void eFace::DebugPlot(PD_Color c){
  if (id>=0 && faces[id]!=this)
    tERR_ERROR_INT("not in facelist!");

  if (pp_tess_deb && se_debugExt>=0){
    int x=int(.33333*p[0]->x+.333333*p[1]->x+.333333*p[2]->x);
    int y=int(.33333*p[0]->y+.333333*p[1]->y+.333333*p[2]->y);
    
    PD_CircleFilled (DoubleBuffer,
		     se_X_ToScreen(x), se_Y_ToScreen(y),
		     3,1,c);
    
    if (se_debugExt) PD_ShowDoubleBuffer();
  }
}
#endif

eFace::eFace (eEdge *e1,eEdge *e2,eEdge *e3):id(-1){
  int i;

  e[0]=e1;e[1]=e2;e[2]=e3;

  // find endpoints of edges

  for(i=0;i<=2;i++){
    p[i]=NULL;
    for(int j=0;j<=1;j++)
      for(int k=0;k<=1;k++)
	if(e[se_Left(i)]->p[j]==e[se_Right(i)]->p[k])
	  p[i]=e[se_Left(i)]->p[j];
    if (!p[i])
      tERR_ERROR_INT("Edges " << *e[se_Left(i)] << " and " << *e[se_Right(i)]  << 
	    " don't have a common endpoint!");

    p[i]->AddNeighbour(this);
  }

  for(i=0;i<=2;i++)
    e[i]->AddNeighbour(this);

  for(i=MAX_VIEWERS-1;i>=0;i--){
    visHeight[i]=0;
  }

  faces.Add(this,id);


#ifdef DEBUG 
  // if (pp_out) DebugPlot(PD_CreateColor(DoubleBuffer,0x90,0x90,0x90));
  
  bool degenerate=true;
  for(i = 2; i>=0 ; i--)
    if (fabs(LeftVec(i)*RightVec(i))>
	.1*EPS*se_EstimatedRangeOfMult(LeftVec(i),RightVec(i)))
      degenerate = false;

  if (degenerate){
    cerr << "Face " << *this << " is degenerate!\n";
    st_Breakpoint();
  }
  
  if (LeftVec(0)*RightVec(0)<0){
    cerr << "Face " << *this << " has wrong orientation!\n";
    st_Breakpoint();
    
    eEdge *s1=e[1];
    e[1]=e[2];
    e[2]=s1;

    ePoint *s2=p[1];
    p[1]=p[2];
    p[2]=s2;
  }
  if (LeftVec(0)*RightVec(0)<0)
    cerr << "Face " << *this << " still has wrong orientation!\n";
#endif

  /*
  // find contained gameObjects
  for(i=eGameObject::gameObjects.Len()-1;i>=0;i--){
    eGameObject *go=eGameObject::gameObjects(i);
    if (go->currentFace==NULL &&
	IsInside(go->pos) &&
	IsInside(go->pos + go->dir*.001))  //improve test!
      go->currentFace=this;
  */
}



eFace::~eFace(){
  int i;

  // unregister gameObjects linked to this eFace
  for(i=eGameObject::gameObjects.Len()-1;i>=0;i--)
    if (eGameObject::gameObjects(i)->currentFace==this)
      eGameObject::gameObjects(i)->currentFace=NULL;


#ifdef POWERPAK_DEB
  if (pp_out)
    DebugPlot(PD_CreateColor(DoubleBuffer,0x0,0x0,0x0));
#endif

  for(i=0;i<=2;i++){
    if (p[i])
      p[i]->RemoveNeighbour(this);
    if (e[i])
      e[i]->RemoveNeighbour(this);
  }
  for(i=0;i<=2;i++){
    p[i]=NULL;
    e[i]=NULL;
  }

  faces.Remove(this,id);

  tCHECK_DEST;
}


int eFace::FindPoint(const ePoint *P) const{
  int ret=-1;
  for(int i=2;i>=0;i--)
    if (p[i]==P){
      ret=i;
      i=-1;
    }
  return ret;
}
int eFace::FindEdge(const eEdge *E) const{
  int ret=-1;

  for(int i=2;i>=0;i--)
    if (e[i]==E){
      ret=i;
      i=-1;
    }
  return ret;

  }




bool eFace::IsInside(const eCoord &c){
  for(int i=0;i<=2;i++){
    eCoord ed=*p[se_Left(i)]-*p[i];
    eCoord cc=c-*p[i];
    if (ed*cc<-EPS)
      return 0;
  }
  return 1;
}


void eFace::UpdateVisAll(int num){
  while (num>0){
    for(int i=NumberOfCameras()-1;i>=0;i--){
        // go to next on_the_eEdge_of_vision - eEdge:
      //cerr << "nonurgent eFace!\n";
      
      if((--s_currentCheckVis[i])<0 ) 
	s_currentCheckVis[i]=faces.Len()-1;
      if (s_currentCheckVis[i]>=faces.Len())
	s_currentCheckVis[i]=faces.Len()-1;
      
      if (s_currentCheckVis[i]>=0)
	faces(s_currentCheckVis[i])->UpdateVis(i);
    }
    num--;
  }
}

  // visibility by viewer i
void eFace::SetVisHeight(int i,REAL h){
  // is the viewer inside? then, the eFace is surely visible.
  if (IsInside(CameraPos(i)))
    h=0;


  if (visHeight[i]!=h){
    visHeight[i]=h;
    for(int j=0;j<=2;j++)
      if (e[j]->UpdateVis(i))
	e[j]->MarkCheckVis(i);
  }
}

void eFace::UpdateVis(int i){

  // check all its edges
  REAL h=10000000;
  
  for(int j=0;j<=2;j++){
    eEdge *e_check=e[j];
    // if e_check is transparent
    eFace *nearer=e_check->NearerToViewer(i);
    if (nearer && nearer!=this){
      REAL oh=nearer->visHeight[i];
      if (e_check->wall && e_check->wall->BlockHeight()>oh)
	oh=e_check->wall->BlockHeight();
      if (oh<h) h=oh;
    }
  }

  if (h>1000000) h=0;
  
  SetVisHeight(i,h);

#ifdef POWERPAK_DEB
  for(int i=0;i<=2;i++)
    e[i]->DebugPlot(PD_CreateColor(DoubleBuffer,0,100,0));
#endif
}


void eFace::Clear(){
  for (int i=faces.Len()-1; i>=0; i--)
    delete faces[i];
}



#define glen 320.0
#define maxi 5
#define maxj 5

/*
void se_CreateGrid(){
  int d=se_debugExt;
  se_debugExt=0;

  ePoint *x[maxi+2][maxj+2];
  eEdge  *e[maxi+2][maxj+2][3];

  int i,j;

  for(i=0;i<maxi+2;i++)
    for(j=0;j<maxj+2;j++)
      x[i][j]=tNEW() ePoint(-100+i*glen-j*glen/2,-100+j*glen*.7);

#ifdef DEBUG
  se_CheckGrid(false);
#endif
    

#ifdef POWERPAK_DEB
  if (d && pp_out) PD_ShowDoubleBuffer();
#endif
  
  for(i=0;i<maxi+1;i++)
    for(j=0;j<maxj+1;j++){
      e[i][j][0]=tNEW() eEdge(x[i][j],x[i+1][j]);
#ifdef DEBUG
  se_CheckGrid(false);
#endif
      e[i][j][1]=tNEW() eEdge(x[i][j],x[i+1][j+1]);
#ifdef DEBUG
  se_CheckGrid(false);
#endif
      e[i][j][2]=tNEW() eEdge(x[i][j],x[i][j+1]);
#ifdef DEBUG
  se_CheckGrid(false);
#endif
    }


#ifdef DEBUG
  se_CheckGrid();
#endif

#ifdef POWERPAK_DEB
  if (d && pp_out) PD_ShowDoubleBuffer();
#endif

  for(i=0;i<maxi;i++)
    for(j=0;j<maxj;j++){
      tNEW() eFace(e[i][j][0],e[i+1][j][2],e[i][j][1]);
      tNEW() eFace(e[i][j][1],e[i][j+1][0],e[i][j][2]);
    }


#ifdef DEBUG
  se_CheckGrid();
#endif

#ifdef POWERPAK_DEB
  if (d && pp_out) PD_ShowDoubleBuffer();
#endif

  se_debugExt=d;

}
*/


eCoord  base(20,100);
static ePoint *A=0,*B=0,*C=0;
static eEdge  *a=0,*b=0,*c=0;
REAL   max_Norm_squared=0;

void se_CreateGrid(){
  base=eCoord(20,100);

  A=tNEW(ePoint) (base.Turn(1,0));
  B=tNEW(ePoint) (base.Turn(-.5,.7));
  C=tNEW(ePoint) (base.Turn(-.5,-.7));
  
  a=tNEW(eEdge) (B,C,tNEW(eWall));
  b=tNEW(eEdge) (C,A,tNEW(eWall));
  c=tNEW(eEdge) (A,B,tNEW(eWall));

  tNEW(eFace) (a,b,c);

  max_Norm_squared=base.Norm_squared();
}


void se_ClearGrid(){
  base=eCoord(0,100);
  max_Norm_squared=0;

  eFace::Clear();
  eEdge::Clear();
  ePoint::Clear();
  
  A=B=C=NULL;
  a=b=c=NULL;
}

void se_GridRange(REAL Norm_squared){
  if (A==NULL)
    tERR_ERROR_INT("Wanted to insert something into the nonexistant grid!");

  while (Norm_squared > max_Norm_squared *.1){
    base = base.Turn(-4,0);
    max_Norm_squared=base.Norm_squared();
    
    delete a->Wall();
    delete b->Wall();
    delete c->Wall();
    
    ePoint *newA=tNEW(ePoint) (base.Turn(1,0));
    ePoint *newB=tNEW(ePoint) (base.Turn(-.5,.7));
    ePoint *newC=tNEW(ePoint) (base.Turn(-.5,-.7));
  
    eEdge *newa=tNEW(eEdge) (newB,newC,tNEW(eWall));
    eEdge *newb=tNEW(eEdge) (newC,newA,tNEW(eWall));
    eEdge *newc=tNEW(eEdge) (newA,newB,tNEW(eWall));
    
    eEdge *AnewB=tNEW(eEdge) (A,newB,NULL);
    eEdge *AnewC=tNEW(eEdge) (A,newC,NULL);
    eEdge *BnewA=tNEW(eEdge) (B,newA,NULL);
    eEdge *BnewC=tNEW(eEdge) (B,newC,NULL);
    eEdge *CnewA=tNEW(eEdge) (C,newA,NULL);
    eEdge *CnewB=tNEW(eEdge) (C,newB,NULL);
    
    tNEW(eFace) (a,BnewA,CnewA);
    tNEW(eFace) (b,CnewB,AnewB);
    tNEW(eFace) (c,AnewC,BnewC);

    tNEW(eFace) (newa,AnewC,AnewB);
    tNEW(eFace) (newb,BnewA,BnewC);
    tNEW(eFace) (newc,CnewB,CnewA);

    a=newa;
    b=newb;
    c=newc;
    
    A=newA;
    B=newB;
    C=newC;
  }
}


void se_ResetVisibles(int ){
  // cleanup
  
  //eFace::SetVisHeightAll(viewer,0);
  //eEdge::MarkCheckVisAll(viewer);

}

void eEdge::MarkCheckVisAll(int viewer){
  for(int i=edges.Len()-1;i>=0;i--)
    edges[i]->MarkCheckVis(viewer);
}


void eFace::SetVisHeightAll(int viewer,REAL h){
  for(int i=faces.Len()-1;i>=0;i--)
    faces[i]->SetVisHeight(viewer,h);
}


