/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#ifndef ArmageTron_MENU_H
#define ArmageTron_MENU_H

#include "rFont.h"
#include "tList.h"
#include "tString.h"
#include "tCallback.h"

#include "rSDL.h"
#ifndef DEDICATED
#include <SDL_events.h>
#endif

#include "defs.h"

class uMenuItem;

class uMenu{
  friend class uMenuItem;
  
  List<uMenuItem>      items;
  static FUNCPTR       idle;

  REAL menuTop;
  REAL menuBot;
  REAL yOffset;

  bool                 exitFlag;

  REAL                 spaceBelow;
  REAL                 center;

  REAL YPos(int num);
public:
  static bool          wrap;
  static bool          quickexit;

  tString              title;

  FUNCPTR IdleFunc(){return idle;}
  static void SetIdle(FUNCPTR idle_func) {idle=idle_func;}
  void SetCenter(REAL c) {center=c;}
  void SetTop(REAL t) {menuTop=t;}
  void SetBot(REAL b) {menuBot=b;spaceBelow=1+menuBot;}

  uMenu(const char *t,bool exit_item=true);
  ~uMenu();
  
  // enters the menu; calls idle_func bef
  void Enter();
  
  void ReverseItems();

  // paints a nice background
  static void GenericBackground();
  
  // marks the menu for exit
  void Exit();

  void RequestSpaceBelow(REAL x){
    if (spaceBelow<x)
      spaceBelow=x;
  }
};


// *****************************************************

class uMenuItem{
  friend class uMenu;

  int id;
  uMenuItem(){};
protected:
  uMenu  *menu;
  tString helpText;
  
  void DisplayText(REAL x,REAL y,const char *text,bool selected,
		    REAL alpha=1,int center=0,int cursor=0,int cursorPos=0);
  void DisplayTextSpecial(REAL x,REAL y,const char *text,bool selected,
			    REAL alpha=1,int center=0);
public:
  uMenuItem(uMenu *M,const char *help)
    :id(-1),menu(M),helpText(help){
    menu->items.Add(this,id);
  }

  virtual ~uMenuItem(){menu->items.Remove(this,id);}

  virtual tString Help(){return helpText;}
  
  // displays the menuitem at position x,y. set selected to true
  // if the item is currently under the cursor
  virtual void Render(REAL ,REAL ,REAL =1,bool =0){};

   virtual void RenderBackground(){
    menu->GenericBackground();
   };
  
  // if the user presses left/right on menuitem
  virtual void LeftRight(int ){}; //lr=-1:left lr=+1: right
  virtual void LeftRightRelease(){}; 
  
  virtual void Enter(){}; // if the user presses enter/space on menu
  
  virtual bool Event(SDL_Event &){return false;} // if the key c is
  // pressed,mouse moved ... 
  // while menuitem is active; return value: key was used

  virtual REAL SpaceRight(){return 0;}
};


// *****************************************************
// Menu Exit
// *****************************************************

class uMenuItemExit:public uMenuItem{
  char *t;
public:
  uMenuItemExit(uMenu *M,
		char *text="Exit Menu",
		char *help="Exits this menu and returns to the previous level")
    :uMenuItem(M,help)
    ,t(text){}
  
  // displays the menuitem at position x,y. set selected to true
  // if the item is currently under the cursor
  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0){
    DisplayTextSpecial(x,y,t,selected,alpha);
  };

  virtual void Enter(){menu->Exit();} 
  // if the user presses enter/space on menu
};

// *****************************************************
// Selection
// *****************************************************

template<class T> class uSelectItem{
public:
  int id;

  tString description;
  tString helpText;
  T value;

  uSelectItem(const char *desc,const char *help,T val)
    :id(-1),description(desc),helpText(help),value(val){}
};

template<class T> class uMenuItemSelection:public uMenuItem{
protected:
  List<uSelectItem<T> > choices;
  tString               title;
  int                   select;
  T *                   target;
public:
  uMenuItemSelection(uMenu *m,
				const char *tit,const char *help,
				T &targ)
    :uMenuItem(m,help),title(tit),select(0),target(&targ){}

  ~uMenuItemSelection(){
    for(int i=choices.Len()-1;i>=0;i--){
      uSelectItem<T> *x=choices(i);
      choices.Remove(x,x->id);
      delete x;
    }
  }

  void NewChoice(uSelectItem<T> *c){
    choices.Add(c,c->id);
  } 

  void NewChoice(const char *desc,const char *help,T val){
    uSelectItem<T> *x=new uSelectItem<T>(desc,help,val);
    NewChoice(x);
  } 

  virtual void LeftRight(int lr){
    select+=lr;
    if(select>=choices.Len())
      select=choices.Len()-1;
    if(select<0)
      select=0;
    if (choices.Len())
      *target=choices(select)->value;
  }
  
  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0){
    for(int i=choices.Len()-1;i>=0;i--)
      if (choices(i)->value==*target)
	select=i;
    
    DisplayText(REAL(x-.02),y,title,selected,alpha,1);
    if (choices.Len()>0)
      DisplayText(REAL(x+.02),y,choices(select)->description,selected,alpha,-1);
  }

  virtual tString Help(){return helpText+choices(select)->helpText;}

};

template<class T> class uSelectEntry{
 public:
  uSelectEntry(uMenuItemSelection<T> &sel,
	       const char *desc,
	       const char *help,T val){
    sel.NewChoice(desc,help,val);
  }
};

// *****************************************
//               Toggle
// *****************************************

class uMenuItemToggle: public uMenuItemSelection<bool>{
  void NewChoice(uSelectItem<bool> *c);
  void NewChoice(const char *desc,bool val);
public:
  uMenuItemToggle(uMenu *m,const char *tit,
			     const char *help,bool &targ);
  ~uMenuItemToggle();

  virtual void LeftRight(int);
  virtual void Enter();
};


// *****************************************
//               Integer Choose
// *****************************************

class uMenuItemInt:public uMenuItem{
protected:
  tString title;
  int &target;
  int Min,Max;
  int Step;
public:
  uMenuItemInt(uMenu *m,const char *tit,
			      const char *help,int &targ,
			int mi,int ma,int step=1);
  ~uMenuItemInt(){};

  virtual void LeftRight(int);

  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0);
};




// *****************************************************
//  String query
// *****************************************************


class uMenuItemString: uMenuItem{
protected:
  tString  description;
  tString *content;
  int      cursorPos;
public:
  uMenuItemString(uMenu *M,const char *descr,
			     const char *help,tString &c);
  virtual ~uMenuItemString(){}

  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0);

  virtual bool Event(SDL_Event &e);

  uMenu *MyMenu(){return menu;}
};


// *****************************************************
//  Submenu
// *****************************************************


class uMenuItemSubmenu: uMenuItem{
  uMenu *submenu;
public:
  uMenuItemSubmenu(uMenu *M,uMenu *s,
			      const char *help);
  virtual ~uMenuItemSubmenu(){}

  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0);

  virtual void Enter();
};


// *****************************************************
//  Execute function
// *****************************************************


class uMenuItemFunction: uMenuItem{
  FUNCPTR  func;
  tString  name;
public:
  uMenuItemFunction(uMenu *M,const char *name,
			       const char *help,FUNCPTR f);

  virtual ~uMenuItemFunction(){}

  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0);

  virtual void Enter();
};


class uMenuItemFunctionInt: uMenuItem{
  INTFUNCPTR  func;
  int         arg;
  tString     name;
public:
  uMenuItemFunctionInt(uMenu *M,const char *name,
				  const char *help,INTFUNCPTR f,int arg);
  //uMenuItemFunctionInt(uMenu *M,const char *name);
  virtual ~uMenuItemFunctionInt(){}

  virtual void Render(REAL x,REAL y,REAL alpha=1,bool selected=0);

  virtual void Enter();
};

// *****************************************************
// Menu Enter/Leave-Callback
// *****************************************************

class uCallbackMenuEnter: public tCallback{
 public:
  uCallbackMenuEnter(VOIDFUNC *f);
  static void MenuEnter();
};

class uCallbackMenuLeave: public tCallback{
 public:
  uCallbackMenuLeave(VOIDFUNC *f);
  static void MenuLeave();
};

class uCallbackMenuBackground: public tCallback{
 public:
  uCallbackMenuBackground(VOIDFUNC *f);
  static void MenuBackground();
};


#endif

